#ifndef KMEDIAN_H
#define KMEDIAN_H

#include <functional>

#include "Metric.hpp"
#include "Point.hpp"
#include "WeightedPoint.hpp"

/**
 * @brief k-median evaluation for Point objects
 */
class KMedian
{
public:
	KMedian(std::function<Metric<Point>*()> createMetric);
	~KMedian();
	
	/**
	 * @brief 1-median cost
	 */
	template<typename ForwardIterator>
	double cost(ForwardIterator first, ForwardIterator last, Point Center);
	
	/**
	 * @brief k-median cost
	 */
	template<typename ForwardIteratorPoint, typename ForwardIteratorCenter>
	double cost(ForwardIteratorPoint beginP, ForwardIteratorPoint endP, ForwardIteratorCenter beginC, ForwardIteratorCenter endC);
	
	/**
	 * @brief 1-median weighted cost
	 */
	template<typename ForwardIterator>
	double weightedCost(ForwardIterator first, ForwardIterator last, Point Center);
	
	/**
	 * @brief k-median weighted cost
	 */
	template<typename ForwardIteratorPoint, typename ForwardIteratorCenter>
	double weightedCost(ForwardIteratorPoint beginP, ForwardIteratorPoint endP, ForwardIteratorCenter beginC, ForwardIteratorCenter endC);
	
private:
	Metric<Point>* dist;
};

template<typename ForwardIterator>
double KMedian::cost(ForwardIterator first, ForwardIterator last, Point center)
{	
	double sum = 0;
	while(first != last)
	{
		sum += dist->distance(toPointer(*first), center);
		++first;
	}
	return sum;
}

template<typename ForwardIteratorPoint, typename ForwardIteratorCenter>
double KMedian::cost(ForwardIteratorPoint beginP, ForwardIteratorPoint endP, ForwardIteratorCenter beginC, ForwardIteratorCenter endC)
{
	double sum = 0;
	for(auto itP = beginP; itP != endP; ++itP)
	{
		double minCost = 0;
		for(auto itC = beginC; itC != endC; ++itC)
		{
			double tmpCost = dist->distance(toPointer(*itP), toPointer(*itC));
			if(tmpCost < minCost || itC == beginC)
				minCost = tmpCost;
		}
		sum += minCost;
	}
	return sum;
}

template<typename ForwardIterator>
double KMedian::weightedCost(ForwardIterator first, ForwardIterator last, Point center)
{	
	double sum = 0;
	while(first != last)
	{
		sum += first->getWeight() * dist->distance(toPointer(*first), center);
		++first;
	}
	return sum;
}

template<typename ForwardIteratorPoint, typename ForwardIteratorCenter>
double KMedian::weightedCost(ForwardIteratorPoint beginP, ForwardIteratorPoint endP, ForwardIteratorCenter beginC, ForwardIteratorCenter endC)
{
	double sum = 0;
	for(auto itP = beginP; itP != endP; ++itP)
	{
		double minCost = 0;
		for(auto itC = beginC; itC != endC; ++itC)
		{
			double tmpCost = toPointer(*itP)->getWeight() * dist->distance(toPointer(*itP), toPointer(*itC));
			if(tmpCost < minCost || itC == beginC)
				minCost = tmpCost;
		}
		sum += minCost;
	}
	return sum;
}

#endif