#include <memory>
#include <iostream>
#include <sstream>
#include <iomanip>
#include <vector>
#include <iterator>
#include <fstream>

#ifdef ERRORHANDLER
#include <cstdlib>
#include <stdexcept>
#include <execinfo.h>
#endif

#include <boost/algorithm/string.hpp>

#include "../probi-source/FastCoreset.hpp"
#include "../probi-source/WeightedPoint.hpp"
#include "../probi-source/Point.hpp"
#include "../probi-source/ProbabilisticPoint.hpp"
#include "../probi-source/EuclideanMetric.hpp"
#include "../probi-source/EuclideanNorm.hpp"
#include "../probi-source/EuclideanSquaredMetric.hpp"
#include "../probi-source/EuclideanSquaredNorm.hpp"

#include "common.hpp"

using namespace std;

int main(int argc, char** argv)
{
#ifdef ERRORHANDLER
    std::set_terminate(handler);
#endif

    if (argc < 7)
    {
        std::cout << "Usage: k splitChar fastCoresetSampleSize numOfPointsToMerge bucketSize seed file" << std::endl;
        std::cout << "e.g. : 10 S 2000 10 4000 28934987 file.dat" << std::endl;
        return 1;
    }

    // k argument
    int k = atoi(argv[1]);
    // SplitChar argument
    char splitChar = *argv[2];
    if (splitChar == 'S')
        splitChar = ' ';
    std::stringstream splitStream;
    splitStream << splitChar;
    std::string splitSeq(splitStream.str());
    // FastCoresetSampleSize argument
    int fastCoresetSampleSize = atoi(argv[3]);
    // NumOfPointsToMerge  argument
    int numOfPointsToMerge = atoi(argv[4]);
    // BucketSize argument
    int bucketSize = atoi(argv[5]);
    // Random seed
    std::mt19937* random = Randomness::getMT19937();
    new (random) std::mt19937(atoi(argv[6]));
    // Input file
    std::fstream filestr(argv[7], std::fstream::in);

#ifdef KMEANS
    FastCoreset fastCoreset([]()
    {
        return new EuclideanSquaredMetric();
    }, []()
    {
        return new EuclideanSquaredNorm();
    });
#else
    FastCoreset fastCoreset([]()
    {
        return new EuclideanMetric();
    }, []()
    {
        return new EuclideanNorm();
    });
#endif

    fastCoreset.setK(k);
    fastCoreset.setAllSamplesSize(fastCoresetSampleSize);
    DatastreamCoreset dsCoreset(&fastCoreset, bucketSize);

    std::vector<ProbabilisticPoint> ppoints;
    double prob = 1 / double(numOfPointsToMerge);

    // Data dimension
    int dim = 0;
    std::function<bool(ProbabilisticPoint) > determineDimension = [&dim] (ProbabilisticPoint pp)
    {
        dim = pp[0].getDimension();
        return false;
    };
    filestr.clear();
    filestr.seekg(0);
    processStream(filestr, determineDimension, splitSeq, numOfPointsToMerge, prob, false);

    // Coreset construction
    std::function<bool(ProbabilisticPoint) > processProbPoint = [&dsCoreset] (ProbabilisticPoint pp)
    {
        dsCoreset << pp;
        return true;
    };
    filestr.clear();
    filestr.seekg(0);
    processStream(filestr, processProbPoint, splitSeq, numOfPointsToMerge, prob, false);

    std::vector<ProbabilisticPoint> ds_coreset;
    ds_coreset = *dsCoreset.assemble();

    // Output
    for (int i = 0; i < ds_coreset.size(); ++i)
    {
        ProbabilisticPoint& pp = ds_coreset[i];
        std::cout << pp.getWeight() << std::endl;
        for (int j = 0; j < pp.getSizeOfDistribution(); ++j)
        {
            WeightedPoint& wp = pp[j];
            for (int k = 0; k < wp.getDimension(); ++k)
            {
                std::cout << wp[k];
                if (k < wp.getDimension() - 1)
                    std::cout << ",";
            }
            std::cout << "\n";
        }
    }

    return 0;
}

